using System;
using System.Data;
using System.Linq;
using System.Linq.Dynamic.Core;
using System.Collections.Generic;
using System.Threading.Tasks;
using System.Text.Encodings.Web;
using Microsoft.AspNetCore.Components;
using Microsoft.Data.SqlClient;
using Microsoft.EntityFrameworkCore;
using Radzen;

using Net8Auto.Server.Data;

namespace Net8Auto.Server
{
    public partial class RadzenSampleService
    {
        RadzenSampleContext Context
        {
           get
           {
             return this.context;
           }
        }

        private readonly RadzenSampleContext context;
        private readonly NavigationManager navigationManager;

        public RadzenSampleService(RadzenSampleContext context, NavigationManager navigationManager)
        {
            this.context = context;
            this.navigationManager = navigationManager;
        }

        public void Reset() => Context.ChangeTracker.Entries().Where(e => e.Entity != null).ToList().ForEach(e => e.State = EntityState.Detached);

        public void ApplyQuery<T>(ref IQueryable<T> items, Query query = null)
        {
            if (query != null)
            {
                if (!string.IsNullOrEmpty(query.Filter))
                {
                    if (query.FilterParameters != null)
                    {
                        items = items.Where(query.Filter, query.FilterParameters);
                    }
                    else
                    {
                        items = items.Where(query.Filter);
                    }
                }

                if (!string.IsNullOrEmpty(query.OrderBy))
                {
                    items = items.OrderBy(query.OrderBy);
                }

                if (query.Skip.HasValue)
                {
                    items = items.Skip(query.Skip.Value);
                }

                if (query.Top.HasValue)
                {
                    items = items.Take(query.Top.Value);
                }
            }
        }


        public async Task ExportOrderDetailsToExcel(Query query = null, string fileName = null)
        {
            navigationManager.NavigateTo(query != null ? query.ToUrl($"export/radzensample/orderdetails/excel(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')") : $"export/radzensample/orderdetails/excel(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')", true);
        }

        public async Task ExportOrderDetailsToCSV(Query query = null, string fileName = null)
        {
            navigationManager.NavigateTo(query != null ? query.ToUrl($"export/radzensample/orderdetails/csv(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')") : $"export/radzensample/orderdetails/csv(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')", true);
        }

        partial void OnOrderDetailsRead(ref IQueryable<Net8Auto.Server.Models.RadzenSample.OrderDetail> items);

        public async Task<IQueryable<Net8Auto.Server.Models.RadzenSample.OrderDetail>> GetOrderDetails(Query query = null)
        {
            var items = Context.OrderDetails.AsQueryable();

            items = items.Include(i => i.Order);
            items = items.Include(i => i.Product);

            if (query != null)
            {
                if (!string.IsNullOrEmpty(query.Expand))
                {
                    var propertiesToExpand = query.Expand.Split(',');
                    foreach(var p in propertiesToExpand)
                    {
                        items = items.Include(p.Trim());
                    }
                }

                ApplyQuery(ref items, query);
            }

            OnOrderDetailsRead(ref items);

            return await Task.FromResult(items);
        }

        partial void OnOrderDetailGet(Net8Auto.Server.Models.RadzenSample.OrderDetail item);
        partial void OnGetOrderDetailById(ref IQueryable<Net8Auto.Server.Models.RadzenSample.OrderDetail> items);


        public async Task<Net8Auto.Server.Models.RadzenSample.OrderDetail> GetOrderDetailById(int id)
        {
            var items = Context.OrderDetails
                              .AsNoTracking()
                              .Where(i => i.Id == id);

            items = items.Include(i => i.Order);
            items = items.Include(i => i.Product);
 
            OnGetOrderDetailById(ref items);

            var itemToReturn = items.FirstOrDefault();

            OnOrderDetailGet(itemToReturn);

            return await Task.FromResult(itemToReturn);
        }

        partial void OnOrderDetailCreated(Net8Auto.Server.Models.RadzenSample.OrderDetail item);
        partial void OnAfterOrderDetailCreated(Net8Auto.Server.Models.RadzenSample.OrderDetail item);

        public async Task<Net8Auto.Server.Models.RadzenSample.OrderDetail> CreateOrderDetail(Net8Auto.Server.Models.RadzenSample.OrderDetail orderdetail)
        {
            OnOrderDetailCreated(orderdetail);

            var existingItem = Context.OrderDetails
                              .Where(i => i.Id == orderdetail.Id)
                              .FirstOrDefault();

            if (existingItem != null)
            {
               throw new Exception("Item already available");
            }            

            try
            {
                Context.OrderDetails.Add(orderdetail);
                Context.SaveChanges();
            }
            catch
            {
                Context.Entry(orderdetail).State = EntityState.Detached;
                throw;
            }

            OnAfterOrderDetailCreated(orderdetail);

            return orderdetail;
        }

        public async Task<Net8Auto.Server.Models.RadzenSample.OrderDetail> CancelOrderDetailChanges(Net8Auto.Server.Models.RadzenSample.OrderDetail item)
        {
            var entityToCancel = Context.Entry(item);
            if (entityToCancel.State == EntityState.Modified)
            {
              entityToCancel.CurrentValues.SetValues(entityToCancel.OriginalValues);
              entityToCancel.State = EntityState.Unchanged;
            }

            return item;
        }

        partial void OnOrderDetailUpdated(Net8Auto.Server.Models.RadzenSample.OrderDetail item);
        partial void OnAfterOrderDetailUpdated(Net8Auto.Server.Models.RadzenSample.OrderDetail item);

        public async Task<Net8Auto.Server.Models.RadzenSample.OrderDetail> UpdateOrderDetail(int id, Net8Auto.Server.Models.RadzenSample.OrderDetail orderdetail)
        {
            OnOrderDetailUpdated(orderdetail);

            var itemToUpdate = Context.OrderDetails
                              .Where(i => i.Id == orderdetail.Id)
                              .FirstOrDefault();

            if (itemToUpdate == null)
            {
               throw new Exception("Item no longer available");
            }
                
            var entryToUpdate = Context.Entry(itemToUpdate);
            entryToUpdate.CurrentValues.SetValues(orderdetail);
            entryToUpdate.State = EntityState.Modified;

            Context.SaveChanges();

            OnAfterOrderDetailUpdated(orderdetail);

            return orderdetail;
        }

        partial void OnOrderDetailDeleted(Net8Auto.Server.Models.RadzenSample.OrderDetail item);
        partial void OnAfterOrderDetailDeleted(Net8Auto.Server.Models.RadzenSample.OrderDetail item);

        public async Task<Net8Auto.Server.Models.RadzenSample.OrderDetail> DeleteOrderDetail(int id)
        {
            var itemToDelete = Context.OrderDetails
                              .Where(i => i.Id == id)
                              .FirstOrDefault();

            if (itemToDelete == null)
            {
               throw new Exception("Item no longer available");
            }

            OnOrderDetailDeleted(itemToDelete);


            Context.OrderDetails.Remove(itemToDelete);

            try
            {
                Context.SaveChanges();
            }
            catch
            {
                Context.Entry(itemToDelete).State = EntityState.Unchanged;
                throw;
            }

            OnAfterOrderDetailDeleted(itemToDelete);

            return itemToDelete;
        }
    
        public async Task ExportOrdersToExcel(Query query = null, string fileName = null)
        {
            navigationManager.NavigateTo(query != null ? query.ToUrl($"export/radzensample/orders/excel(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')") : $"export/radzensample/orders/excel(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')", true);
        }

        public async Task ExportOrdersToCSV(Query query = null, string fileName = null)
        {
            navigationManager.NavigateTo(query != null ? query.ToUrl($"export/radzensample/orders/csv(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')") : $"export/radzensample/orders/csv(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')", true);
        }

        partial void OnOrdersRead(ref IQueryable<Net8Auto.Server.Models.RadzenSample.Order> items);

        public async Task<IQueryable<Net8Auto.Server.Models.RadzenSample.Order>> GetOrders(Query query = null)
        {
            var items = Context.Orders.AsQueryable();


            if (query != null)
            {
                if (!string.IsNullOrEmpty(query.Expand))
                {
                    var propertiesToExpand = query.Expand.Split(',');
                    foreach(var p in propertiesToExpand)
                    {
                        items = items.Include(p.Trim());
                    }
                }

                ApplyQuery(ref items, query);
            }

            OnOrdersRead(ref items);

            return await Task.FromResult(items);
        }

        partial void OnOrderGet(Net8Auto.Server.Models.RadzenSample.Order item);
        partial void OnGetOrderById(ref IQueryable<Net8Auto.Server.Models.RadzenSample.Order> items);


        public async Task<Net8Auto.Server.Models.RadzenSample.Order> GetOrderById(int id)
        {
            var items = Context.Orders
                              .AsNoTracking()
                              .Where(i => i.Id == id);

 
            OnGetOrderById(ref items);

            var itemToReturn = items.FirstOrDefault();

            OnOrderGet(itemToReturn);

            return await Task.FromResult(itemToReturn);
        }

        partial void OnOrderCreated(Net8Auto.Server.Models.RadzenSample.Order item);
        partial void OnAfterOrderCreated(Net8Auto.Server.Models.RadzenSample.Order item);

        public async Task<Net8Auto.Server.Models.RadzenSample.Order> CreateOrder(Net8Auto.Server.Models.RadzenSample.Order order)
        {
            OnOrderCreated(order);

            var existingItem = Context.Orders
                              .Where(i => i.Id == order.Id)
                              .FirstOrDefault();

            if (existingItem != null)
            {
               throw new Exception("Item already available");
            }            

            try
            {
                Context.Orders.Add(order);
                Context.SaveChanges();
            }
            catch
            {
                Context.Entry(order).State = EntityState.Detached;
                throw;
            }

            OnAfterOrderCreated(order);

            return order;
        }

        public async Task<Net8Auto.Server.Models.RadzenSample.Order> CancelOrderChanges(Net8Auto.Server.Models.RadzenSample.Order item)
        {
            var entityToCancel = Context.Entry(item);
            if (entityToCancel.State == EntityState.Modified)
            {
              entityToCancel.CurrentValues.SetValues(entityToCancel.OriginalValues);
              entityToCancel.State = EntityState.Unchanged;
            }

            return item;
        }

        partial void OnOrderUpdated(Net8Auto.Server.Models.RadzenSample.Order item);
        partial void OnAfterOrderUpdated(Net8Auto.Server.Models.RadzenSample.Order item);

        public async Task<Net8Auto.Server.Models.RadzenSample.Order> UpdateOrder(int id, Net8Auto.Server.Models.RadzenSample.Order order)
        {
            OnOrderUpdated(order);

            var itemToUpdate = Context.Orders
                              .Where(i => i.Id == order.Id)
                              .FirstOrDefault();

            if (itemToUpdate == null)
            {
               throw new Exception("Item no longer available");
            }
                
            var entryToUpdate = Context.Entry(itemToUpdate);
            entryToUpdate.CurrentValues.SetValues(order);
            entryToUpdate.State = EntityState.Modified;

            Context.SaveChanges();

            OnAfterOrderUpdated(order);

            return order;
        }

        partial void OnOrderDeleted(Net8Auto.Server.Models.RadzenSample.Order item);
        partial void OnAfterOrderDeleted(Net8Auto.Server.Models.RadzenSample.Order item);

        public async Task<Net8Auto.Server.Models.RadzenSample.Order> DeleteOrder(int id)
        {
            var itemToDelete = Context.Orders
                              .Where(i => i.Id == id)
                              .Include(i => i.OrderDetails)
                              .FirstOrDefault();

            if (itemToDelete == null)
            {
               throw new Exception("Item no longer available");
            }

            OnOrderDeleted(itemToDelete);


            Context.Orders.Remove(itemToDelete);

            try
            {
                Context.SaveChanges();
            }
            catch
            {
                Context.Entry(itemToDelete).State = EntityState.Unchanged;
                throw;
            }

            OnAfterOrderDeleted(itemToDelete);

            return itemToDelete;
        }
    
        public async Task ExportProductsToExcel(Query query = null, string fileName = null)
        {
            navigationManager.NavigateTo(query != null ? query.ToUrl($"export/radzensample/products/excel(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')") : $"export/radzensample/products/excel(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')", true);
        }

        public async Task ExportProductsToCSV(Query query = null, string fileName = null)
        {
            navigationManager.NavigateTo(query != null ? query.ToUrl($"export/radzensample/products/csv(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')") : $"export/radzensample/products/csv(fileName='{(!string.IsNullOrEmpty(fileName) ? UrlEncoder.Default.Encode(fileName) : "Export")}')", true);
        }

        partial void OnProductsRead(ref IQueryable<Net8Auto.Server.Models.RadzenSample.Product> items);

        public async Task<IQueryable<Net8Auto.Server.Models.RadzenSample.Product>> GetProducts(Query query = null)
        {
            var items = Context.Products.AsQueryable();


            if (query != null)
            {
                if (!string.IsNullOrEmpty(query.Expand))
                {
                    var propertiesToExpand = query.Expand.Split(',');
                    foreach(var p in propertiesToExpand)
                    {
                        items = items.Include(p.Trim());
                    }
                }

                ApplyQuery(ref items, query);
            }

            OnProductsRead(ref items);

            return await Task.FromResult(items);
        }

        partial void OnProductGet(Net8Auto.Server.Models.RadzenSample.Product item);
        partial void OnGetProductById(ref IQueryable<Net8Auto.Server.Models.RadzenSample.Product> items);


        public async Task<Net8Auto.Server.Models.RadzenSample.Product> GetProductById(int id)
        {
            var items = Context.Products
                              .AsNoTracking()
                              .Where(i => i.Id == id);

 
            OnGetProductById(ref items);

            var itemToReturn = items.FirstOrDefault();

            OnProductGet(itemToReturn);

            return await Task.FromResult(itemToReturn);
        }

        partial void OnProductCreated(Net8Auto.Server.Models.RadzenSample.Product item);
        partial void OnAfterProductCreated(Net8Auto.Server.Models.RadzenSample.Product item);

        public async Task<Net8Auto.Server.Models.RadzenSample.Product> CreateProduct(Net8Auto.Server.Models.RadzenSample.Product product)
        {
            OnProductCreated(product);

            var existingItem = Context.Products
                              .Where(i => i.Id == product.Id)
                              .FirstOrDefault();

            if (existingItem != null)
            {
               throw new Exception("Item already available");
            }            

            try
            {
                Context.Products.Add(product);
                Context.SaveChanges();
            }
            catch
            {
                Context.Entry(product).State = EntityState.Detached;
                throw;
            }

            OnAfterProductCreated(product);

            return product;
        }

        public async Task<Net8Auto.Server.Models.RadzenSample.Product> CancelProductChanges(Net8Auto.Server.Models.RadzenSample.Product item)
        {
            var entityToCancel = Context.Entry(item);
            if (entityToCancel.State == EntityState.Modified)
            {
              entityToCancel.CurrentValues.SetValues(entityToCancel.OriginalValues);
              entityToCancel.State = EntityState.Unchanged;
            }

            return item;
        }

        partial void OnProductUpdated(Net8Auto.Server.Models.RadzenSample.Product item);
        partial void OnAfterProductUpdated(Net8Auto.Server.Models.RadzenSample.Product item);

        public async Task<Net8Auto.Server.Models.RadzenSample.Product> UpdateProduct(int id, Net8Auto.Server.Models.RadzenSample.Product product)
        {
            OnProductUpdated(product);

            var itemToUpdate = Context.Products
                              .Where(i => i.Id == product.Id)
                              .FirstOrDefault();

            if (itemToUpdate == null)
            {
               throw new Exception("Item no longer available");
            }
                
            var entryToUpdate = Context.Entry(itemToUpdate);
            entryToUpdate.CurrentValues.SetValues(product);
            entryToUpdate.State = EntityState.Modified;

            Context.SaveChanges();

            OnAfterProductUpdated(product);

            return product;
        }

        partial void OnProductDeleted(Net8Auto.Server.Models.RadzenSample.Product item);
        partial void OnAfterProductDeleted(Net8Auto.Server.Models.RadzenSample.Product item);

        public async Task<Net8Auto.Server.Models.RadzenSample.Product> DeleteProduct(int id)
        {
            var itemToDelete = Context.Products
                              .Where(i => i.Id == id)
                              .Include(i => i.OrderDetails)
                              .FirstOrDefault();

            if (itemToDelete == null)
            {
               throw new Exception("Item no longer available");
            }

            OnProductDeleted(itemToDelete);


            Context.Products.Remove(itemToDelete);

            try
            {
                Context.SaveChanges();
            }
            catch
            {
                Context.Entry(itemToDelete).State = EntityState.Unchanged;
                throw;
            }

            OnAfterProductDeleted(itemToDelete);

            return itemToDelete;
        }
        }
}